/*
 *
 *   OpenChanfix v2.0 
 *
 *   Channel Re-op Service Module for Hybrid 7.1 or later.
 *   Copyright (C) 2003 Thomas Carlsson and Joost Vunderink.
 *   See http://www.garion.org/ocf/ for more information.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */




#include "cf_base.h"

/* ----------------------------------------------------------------- */

void ocf_update_database(void* args)
{
	gather_all_channels();
}

/* ----------------------------------------------------------------- */

void ocf_save_database(void* args)
{
	eventDelete(ocf_save_database, NULL);
	DB_save();
	add_save_database_event();
	ocf_log(OCF_LOG_MAIN, "DBSV Saved database.");
}

/* ----------------------------------------------------------------- */

void ocf_check_opless_channels(void* args)
{
	dlink_node *node_chan;
	struct Channel *chptr;
	int flags, channel_exists, min_clients, dummy;

	/* If autofixing has been disabled, well, forget it. */
	if (!settings_get_bool("enable_autofix")) {
		return;
	}

	/* If there are too many servers split, don't autofix.
	 * Only check if we still have too many servers split, and
	 * report if there has been a change. */
	if (autofix_disabled_due_to_split) {
		autofix_disabled_due_to_split = !check_minimum_servers_linked();
		if (autofix_disabled_due_to_split == 0) {
			/* More servers have linked. We can autofix again. */
			ocf_log(OCF_LOG_MAIN, "AUFX enabled - enough servers.");
			send_notice_to_loggedin(OCF_NOTICE_FLAGS_AUTOFIX,
				"AUTOFIX enabled - enough servers.");
		}
		return;
	} else {
		autofix_disabled_due_to_split = !check_minimum_servers_linked();
		if (autofix_disabled_due_to_split) {
			/* Servers have delinked. Disable autofix. */
			ocf_log(OCF_LOG_MAIN, "AUFX disabled - not enough servers linked.");
			send_notice_to_loggedin(OCF_NOTICE_FLAGS_AUTOFIX,
				"AUTOFIX disabled - not enough servers linked.");
			return;
		}
	}

	/* Get the minimum number of clients that must be present in a channel
	 * before it will be fixed. */
	min_clients = settings_get_int("min_clients");

	/* Now walk through all channels to find the opless ones. */
	DLINK_FOREACH(node_chan, global_channel_list.head) {
		chptr = (struct Channel*)(node_chan->data);
		if (dlink_list_length(&chptr->members) >= min_clients &&
			ocf_is_channel_opless(chptr)) {

			/* Only add this to the autofix list if :
			 * - there are scores on this channel in the database,
			 * - and the channel is not already being fixed,
			 * - and the channel is not blocked. */
			channel_exists = DB_set_read_channel(chptr->chname);
			if (channel_exists && DB_get_op_scores( &dummy, 1 )) {
				flags = DB_channel_get_flag();
				if (flags & OCF_CHANNEL_FLAG_BLOCK || 
						is_being_fixed(chptr->chname)) {
					/* ignore channel */
				} else {
					ocf_log(OCF_LOG_AUTOFIX, "AUFX %s is opless, fixing.",
						chptr->chname);
					send_notice_to_loggedin(OCF_NOTICE_FLAGS_AUTOFIX,
						"AUTOFIX %s is opless, fixing.", chptr->chname);
					add_autofix_channel(chptr->chname);
				}
			}
		}
	}
}

/* ----------------------------------------------------------------- */

void ocf_fix_chanfix_channels(void* args)
{
	dlink_node *ptr, *next_ptr;
	struct chanfix_channel *cfc;
	int channel_fixed;
	time_t now;

	now = time(NULL);

	DLINK_FOREACH_SAFE(ptr, next_ptr, chanfix_channel_list.head) {
		cfc = (struct chanfix_channel*)(ptr->data);
		channel_fixed = 0;

		/* Process stage 1 channels. */
		if (cfc->stage == 1) {
			channel_fixed = ocf_manual_fix_stage_one(cfc);

		/* Process stage 2 channels. */
		} else {
			if (now - cfc->time_previous_attempt < OCF_CHANFIX_INTERVAL) {
				/* do nothing */
			} else {
				channel_fixed = ocf_manual_fix_stage_two(cfc);
			}
		}

		/* If the channel has been fixed, or the fixing time window
		 * has passed, remove it from the list */
		if (channel_fixed ||
			now - cfc->time_fix_started > OCF_CHANFIX_MAXIMUM + OCF_CHANFIX_DELAY) {
			ocf_log(OCF_LOG_CHANFIX, "CHFX Manual fix of \"%s\" complete.",
				cfc->name);
			send_notice_to_loggedin(OCF_NOTICE_FLAGS_CHANFIX,
				"Manual chanfix of \"%s\" complete.", cfc->name);
			del_chanfix_channel(cfc->name);
		}
	}
}

/* ----------------------------------------------------------------- */

void ocf_fix_autofix_channels(void* args)
{
	dlink_node *ptr, *next_ptr;
	struct autofix_channel *afc;
	int channel_fixed;
	time_t now;

	now = time(NULL);

	DLINK_FOREACH_SAFE(ptr, next_ptr, autofix_channel_list.head) {
		afc = (struct autofix_channel*)(ptr->data);
		channel_fixed = 0;

		if (now - afc->time_previous_attempt < OCF_AUTOFIX_INTERVAL) {
			/* do nothing */
		} else {
			channel_fixed = ocf_fix_opless_channel(afc);
		}

		/* If the channel has been fixed, or the fixing time window
		 * has passed, remove it from the list */
		if (channel_fixed ||
			now - afc->time_fix_started > OCF_AUTOFIX_MAXIMUM) {
			ocf_log(OCF_LOG_AUTOFIX, "AUFX Autofix for \"%s\" complete.",
				afc->name);
			send_notice_to_loggedin(OCF_NOTICE_FLAGS_AUTOFIX,
				"Autofix for \"%s\" complete.", afc->name);
			del_autofix_channel(afc->name);
		}
	}
}

/* ----------------------------------------------------------------- */

void add_update_database_event() 
{
	int seconds_since_midnight, next_time_for_event;

	seconds_since_midnight = get_seconds_in_current_day();
	next_time_for_event = 
		seconds_since_midnight + OCF_DATABASE_UPDATE_TIME -
		(seconds_since_midnight % OCF_DATABASE_UPDATE_TIME);

	if (next_time_for_event >= 86400) {
		next_time_for_event = 86400;
	}

	eventAdd("ocf_update_database", &ocf_update_database, NULL, 
		next_time_for_event - seconds_since_midnight);

	ocf_log(OCF_LOG_DEBUG, "database update event, now %d, in %d secs",
			seconds_since_midnight, next_time_for_event - seconds_since_midnight);
}

/* ----------------------------------------------------------------- */

void add_save_database_event() 
{
	int seconds_since_midnight, next_time_for_event;

	seconds_since_midnight = get_seconds_in_current_day();
	next_time_for_event = 
		seconds_since_midnight + OCF_DATABASE_SAVE_TIME -
		(seconds_since_midnight % OCF_DATABASE_SAVE_TIME) +
		OCF_DATABASE_SAVE_OFFSET;

	if (next_time_for_event >= 86400) {
		next_time_for_event = 86400 + OCF_DATABASE_SAVE_OFFSET;
	}

	eventAdd("ocf_save_database", &ocf_save_database, NULL, 
		next_time_for_event - seconds_since_midnight);

	ocf_log(OCF_LOG_DEBUG, "database save event, now %d, in %d secs",
			seconds_since_midnight, next_time_for_event - seconds_since_midnight);
}

/* ----------------------------------------------------------------- */

