/*
 *
 *   OpenChanfix v2.0 
 *
 *   Channel Re-op Service Module for Hybrid 7.1 or later.
 *   Copyright (C) 2003 Thomas Carlsson and Joost Vunderink.
 *   See http://www.garion.org/ocf/ for more information.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



#include "cf_base.h"


/* ----------------------------------------------------------------- */

process_cmd_t process_cmd_functions[] = {
	process_cmd_unknown,
	process_cmd_help,
	process_cmd_score,
	process_cmd_chanfix,
	process_cmd_login,
	process_cmd_logout,
	process_cmd_passwd,
	process_cmd_history,
	process_cmd_info,
	process_cmd_oplist,
	process_cmd_addnote,
	process_cmd_delnote,
	process_cmd_whois,
	process_cmd_addflag,
	process_cmd_delflag,
	process_cmd_addhost,
	process_cmd_delhost,
	process_cmd_adduser,
	process_cmd_deluser,
	process_cmd_chpass,
	process_cmd_who,
	process_cmd_umode,
	process_cmd_status,
	process_cmd_set,
	process_cmd_block,
	process_cmd_unblock,
	process_cmd_rehash,
	process_cmd_check,
	process_cmd_opnicks,
	process_cmd_cscore,
	process_cmd_alert,
	process_cmd_unalert,
};

/* ----------------------------------------------------------------- */

void send_notice_to_loggedin(int flag_required, char *pattern, ...)
{
	va_list args;
	char tmpBuf[IRCLINELEN];
	dlink_node *ptr, *next_ptr;
	struct ocf_loggedin_user *olu;
	int ver_user;

	va_start(args, pattern);
	vsprintf(tmpBuf, pattern, args);
	va_end(args);
	tmpBuf[IRCLINELEN - 1] = 0;

	DLINK_FOREACH_SAFE(ptr, next_ptr, ocf_loggedin_user_list.head) {
		olu = ((struct ocf_loggedin_user*)(ptr->data));
		if (flag_required == OCF_NOTICE_FLAGS_ALL ||
				ocf_user_has_notice_flag(olu->ocf_user_p, flag_required)) {
			/* Make sure the client that currently has nickname olu->nick
			 * is the same as the one that logged in. */
			ver_user = verify_loggedin_user(olu);
			if (ver_user == 1) {
				chanfix_send_privmsg(olu->nick, "%s", tmpBuf);
			} else if (ver_user == 0 ) {
				/* Suppress the message, but don't logout the user. */	
			} else {
				ocf_user_logout(olu->nick);
			}
		}
	}
}

/* ----------------------------------------------------------------- */

void send_notice_to_loggedin_except(struct ocf_loggedin_user *user_p, 
							 int flag_required, char *pattern, ...)
{
	va_list args;
	char tmpBuf[IRCLINELEN];
	dlink_node *ptr, *next_ptr;
	struct ocf_loggedin_user *olu;
	int ver_user;

	va_start(args, pattern);
	vsprintf(tmpBuf, pattern, args);
	va_end(args);
	tmpBuf[IRCLINELEN - 1] = 0;

	DLINK_FOREACH_SAFE(ptr, next_ptr, ocf_loggedin_user_list.head) {
		olu = ((struct ocf_loggedin_user*)(ptr->data));
		if ( (flag_required == OCF_NOTICE_FLAGS_ALL || 
				ocf_user_has_notice_flag(olu->ocf_user_p, flag_required) ) &&
				user_p != olu) {
			/* Make sure the client that currently has nickname olu->nick
			 * is the same as the one that logged in. */
			ver_user = verify_loggedin_user(olu);
			if (ver_user == 1) {
				chanfix_send_privmsg(olu->nick, "%s", tmpBuf);
			} else if (ver_user == 0 ) {
				/* Suppress the message, but don't logout the user. */	
			} else {
				ocf_user_logout(olu->nick);
			}
		}
	}
}

/* ----------------------------------------------------------------- */

void send_notice_to_loggedin_username(const char* username, char *pattern, ...)
{
	va_list args;
	char tmpBuf[IRCLINELEN];
	dlink_node *ptr;
	struct ocf_loggedin_user *olu;
	int ver_user;

	va_start(args, pattern);
	vsprintf(tmpBuf, pattern, args);
	va_end(args);
	tmpBuf[IRCLINELEN - 1] = 0;

	DLINK_FOREACH(ptr, ocf_loggedin_user_list.head) {
		olu = ((struct ocf_loggedin_user*)(ptr->data));
		if (!strcmp(olu->ocf_user_p->name, username)) {
			/* Make sure the client that currently has nickname olu->nick
			 * is the same as the one that logged in. */
			ver_user = verify_loggedin_user(olu);
			if (ver_user == 1) {
				chanfix_send_privmsg(olu->nick, "%s", tmpBuf);
			} else if (ver_user == 0 ) {
				/* Suppress the message, but don't logout the user. */	
			} else {
				ocf_user_logout(olu->nick);
			}
		}
	}
}

/* ----------------------------------------------------------------- */

int process_received_messages(const char* readBuf, int length)
{
	char tmpBuf[READBUF_SIZE];
	struct message_data md;
	int begin = 0;
	int end = 0;
	int num_msgs = 0;

	/* Redundant safety measure: */
//	memset( &md, 0, sizeof( struct message_data ) );

	/* Split the incoming buffer into lines and process each line
	 * separately */
	while (end < length){
		if (readBuf[end] == '\n'){
			memcpy(tmpBuf, &readBuf[begin], end - begin);
			tmpBuf[end - begin - 1] = 0; /* there is \r\n at the end, which is 2 chars*/
			get_message_data(tmpBuf, &md);
			ocf_log(OCF_LOG_DEBUG, "prm: [%d] %s", end-begin-1, tmpBuf);
			process_single_message(&md);
			num_msgs++;
			begin = end + 1;
		}
		end++;
	}

	return num_msgs;
}


/* ----------------------------------------------------------------- */

void process_single_message(struct message_data *md)
{
	switch (md->type) {
	case MSG_UNKNOWN: break;
	case MSG_PING: 
		chanfix->lasttime = CurrentTime;
		ClearPingSent(chanfix);
		break;
	case MSG_PRIVMSG: 
		process_privmsg(md);
		break;
	case MSG_CHANMSG: 
		process_chanmsg(md);
		break;
	case MSG_CTCP: 
		ocf_log(OCF_LOG_DEBUG, "Got a ctcp %s", md->data);
		break;
	case MSG_KILLED:
		process_killed(md);
		break;
	case MSG_RECON_TOO_FAST:
		ocf_log(OCF_LOG_DEBUG, "Trying to reconnect too fast.");
		chanfix = NULL;
		OCF_connect_chanfix_with_delay(OCF_CONNECT_DELAY * 10);
		break;
	case MSG_NUMERIC:
		ocf_log(OCF_LOG_DEBUG, "Got numeric %d.", md->type);
		break;
	}
}

/* ----------------------------------------------------------------- */

void get_message_data(char *msg, struct message_data *md)
{
	md->type = MSG_UNKNOWN;

	if (!strncmp( msg, "PING :", 6 )) {
		md->type = MSG_PING;
		strncpy( md->server, msg + 6, sizeof(md->server) );
		return;
	}

	/* If the first space is followed by 'PRIVMSG #', this is a
	 * PRIVMSG sent to a channel i'm on.
	 * :sabre2th!sabre2th@talking.yoda.is PRIVMSG #carnique :hi C
	 */
	if (strstr(msg, " ") == strstr(msg, " PRIVMSG #")) {
		get_chanmsg_data(msg, md);
		return;
	}
	
	/* If the first space is followed by 'PRIVMSG', this is a
	 * PRIVMSG sent to me. Syntax:
	 * :Garion!garion@chan.fix PRIVMSG C :chanfix #test
	 */
	if (strstr(msg, " ") == strstr(msg, " PRIVMSG ")) {
		get_privmsg_data(msg, md);
		return;
	}

	if (!strncmp(msg, "ERROR :Closing Link:", 20)) {
		md->type = MSG_KILLED;
	}

	if (!strncmp(msg, "ERROR :Trying to reconnect too fast", 35)) {
		md->type = MSG_RECON_TOO_FAST;
	}
	
	if (strstr(msg, " ") == strstr(msg, " 001 ")) {
		md->type = MSG_NUMERIC;
		md->type = 1;
	}
}

/* ----------------------------------------------------------------- */
/*
 *  A safer parser, returns 0 if incomplete/unreliable details were
 *  encountered, 1 if nick only, 3 if nick, user and host were
 *  successfully found.
 */
int parse_userhost_details( char *msg, struct message_data *md )
{
	int c;
	char* m[3] = { md->nick, md->user, md->host };
	int ml[3] = { NICKLEN, USERLEN, HOSTLEN };
	int mc[3] = { 0, 0, 0 };
	char* f;

	/* This check may seem a bit anal, but Mr Murphy loves it! */
	if ( !msg || !md ) return 0;

	for ( c = 0, f = msg; 1; f++ )
	{
		switch ( *f )
		{
			case ':':
			{
				/* Ignore colons */
				break;
			}
			case 0:
			case '\r':
			case '\n':
			{
				/* Something fishy with this string, let's disqualify */
				return 0;
			}
			case '!':
			{
				/* If we ran into the ! while filling the nick element,
				 * we'll move to user
				 */
				if ( c == 0 )
				{
					*m[0] = 0;
					c = 1;
				}
				else
				{
					/* This is unexpected, let's disqualify the whole string */
					return 0;
				}
				break;
			}
			case '@':
			{
				/* If we ran into the @ while filling the user element,
				 * we'll move to host.
				 */
				if ( c == 1 )
				{
					*m[1] = 0;
					c = 2;
				}
				else
				{
					/* This is unexpected, let's disqualify the whole string */
					return 0;
				}
				break;
			}
//			case 0:	/* Uncomment this to make parser more generic */
			case ' ':
			{
				/* Space at the end terminates parse */
				if ( mc[0] )
				{
 					m[c] = 0;
					return c + 1;
				}
				break;
			}
			default:
			{
				/* Let's make sure we're not about to overflow any structures,
				 * in case a compromised ircd sends us bad data.
				 */
				if ( mc[c] >= ml[c] )
				{
					/* Uh oh, going to overflow.. Let's disqualify the whole
					 * parse.
					 */
					*m[c] = 0;
					return 0;
				}

				/* By default, we just fill the active buffer */
				*m[c] = *f;
				m[c]++;
				mc[c]++;
			}
		}
	}
	/* Should never reach this point, but this will shut up compiler warnings */
	return 0;
}

/* ----------------------------------------------------------------- */
/*
 * :Garion!garion@chan.fix PRIVMSG C :chanfix #test
 * or a CTCP which has \001 as first char.
 */
void get_privmsg_data(char *msg, struct message_data *md)
{
	char *cptr;
	struct Client *clptr;
	int uh_parse_results;
	int data_parse_state;
	int i, ctcp_valid;

	int id_name_idx;
	int id_server_idx;
	int id_isnick;
	char id_name[USERLEN + NICKLEN + 1];
	char id_server[HOSTLEN + 1];


	memset( md, 0, sizeof( struct message_data ) );
	md->type = MSG_PRIVMSG;

	uh_parse_results = parse_userhost_details( msg, md );

	if ( uh_parse_results == 1 )
	{
		/* We only got a nickname, let's poll for the user and host */
		clptr = find_client( md->nick );
		if ( clptr )
		{
			/* Note: strncpy does not automatically append a zero at the end
			 * if the length of the src is bigger than the available buffer.
			 * However, since we zeroed this memory structure beforehand, this
			 * SHOULD be safe.
			 */
			strncpy( md->user, clptr->username, USERLEN );
			strncpy( md->host, clptr->host, HOSTLEN );
			strncpy( md->server, clptr->servptr->name, HOSTLEN );
			ocf_log(OCF_LOG_DEBUG, "Got a nick-only message. U@H is %s@%s, server is %s", md->user, md->host, md->server);
		}
		else
		{
			/* We received a message from a nickname which does not exist
			 * in our records.. Fishy indeed. Let's ignore the message.
			 */
			ocf_log(OCF_LOG_DEBUG, "I received a message from non-existant nickname: '%s' (ignored)", msg);
			memset( md, 0, sizeof( struct message_data ) );
			md->type = MSG_UNKNOWN;
			return;
		}
	}
	else
	if ( uh_parse_results == 3 )
	{
		/* We got the nick, user and host, let's validate them to be sure */
		clptr = find_client( md->nick );
		if ( clptr )
		{
			if ( strcasecmp( md->user, clptr->username ) ||
				 strcasecmp( md->host, clptr->host ) )
			{
				/* The username or hostname passed by the PRIVMSG differs from
				 * the local record. Should this ever happen? Let's ignore for now.
				 */
				ocf_log(OCF_LOG_DEBUG, "PRIVMSG user@host mismatches my own record (%s@%s) for nickname: '%s' (ignored)", clptr->username, clptr->host, msg);
				memset( md, 0, sizeof( struct message_data ) );
				md->type = MSG_UNKNOWN;
				return;
			}
			strncpy( md->server, clptr->servptr->name, HOSTLEN );
		}
		else
		{
			/* We received a message from a nickname which does not exist
			 * in our records.. Fishy indeed. Let's ignore the message.
			 */
			ocf_log(OCF_LOG_DEBUG, "I received a message from non-existant nickname: '%s' (ignored)", msg);
			memset( md, 0, sizeof( struct message_data ) );
			md->type = MSG_UNKNOWN;
			return;
		}
	}
	else
	{
		/* Malformed message encountered, let's abort */
		ocf_log(OCF_LOG_DEBUG, "I received a message with malformed nick!user@host: '%s' (aborted parse)", msg);
		memset( md, 0, sizeof( struct message_data ) );
		md->type = MSG_UNKNOWN;
		return;
	}

	/* Now to find what was messaged to me */
	cptr = strstr(msg, " PRIVMSG ");

	if ( !cptr )
	{
		/* Not expected, we'll abort */
		memset( md, 0, sizeof( struct message_data ) );
		md->type = MSG_UNKNOWN;
		return;
	}

	/* Walk 1 + 7 + 1 chars */
	cptr += 9;

	/* Sweep past the recipient field.
	 * Here we will verify that the recipient field is what can reasonably
 	 * be expected. We will also check if this message was sent to 
	 * user@server (safe) rather than just nickname (unsafe). Any masks
	 * will be ignored, and server-wide message recipient processing
	 * aborted altogether.
	 */
	data_parse_state = 0;
	id_name_idx = 0;
	id_server_idx = 0;
	id_isnick = 1;

	memset( id_name, 0, USERLEN + NICKLEN + 1 );
	memset( id_server, 0, HOSTLEN + 1 );

	while ( 1 )
	{
		if ( !*cptr )
		{
			/* Should not null-terminate here, abort */
			memset( md, 0, sizeof( struct message_data ) );
			md->type = MSG_UNKNOWN;
			return;
		}
		else
		if ( *cptr == ' ' )
		{
			switch ( data_parse_state )
			{
				default:
				case 0:
				case 4:
				{
					/* Whitespace before or after recipient, ignore */
					break;
				}
				case 1:
				case 2:
				case 3:
				{
					/* First whitespace after recipient, shift state */
					data_parse_state = 4;

					id_name[id_name_idx] = 0;
					id_server[id_server_idx] = 0;
					break;
				}
			}
		}
		else
		{
			/* Auto-invalidate the message if we encounter unexpected chars */
			if ( *cptr == '$' || *cptr == '!' || *cptr == '\n' )
			{
				memset( md, 0, sizeof( struct message_data ) );
				md->type = MSG_UNKNOWN;
				return;
			}

			if ( data_parse_state == 0 )
			{
				/* Receipient data begins */
				data_parse_state = 1;
			}

			if ( data_parse_state == 1 )
			{
				/* Recipient data continues */
				if ( *cptr == '@' )
				{
					id_isnick = 0;
					id_name[id_name_idx] = 0;
					data_parse_state = 3;
				}
				else
				if ( *cptr == '%' )
				{
					/* There's a mask coming up */
					id_isnick = 0;
					id_name[id_name_idx] = 0;
					data_parse_state = 2;
				}
				else
				{
					id_name[id_name_idx] = *cptr;
					id_name_idx++;
					if ( id_name_idx > ( USERLEN + NICKLEN ) )
					{
						/* Unexpected, let's ignore (we don't care if it goes past
						 * NICKLEN, as we'll only use this string for a string compare)
						 */
						memset( md, 0, sizeof( struct message_data ) );
						md->type = MSG_UNKNOWN;
						return;
					}
				}
			}
			else
			if ( data_parse_state == 2 )
			{
				/* Recipient mask continues (ignored) */
				if ( *cptr == '@' )
				{
					data_parse_state = 3;
				}
			}
			else
			if ( data_parse_state == 3 )
			{
				/* Recipient server continues */
				id_server[id_server_idx] = *cptr;
				id_server_idx++;
				if ( id_server_idx > HOSTLEN )
				{
					/* Unexpected, let's ignore */
					memset( md, 0, sizeof( struct message_data ) );
					md->type = MSG_UNKNOWN;
					return;
				}
			}
			else
			{
				/* Next data field, let's stop processing */
				break;
			}
		}
		cptr++;
	}

	if ( !id_isnick )
	{
		if ( strcasecmp( id_name, chanfix->username ) || strcasecmp( id_server, me.name ) )
		{
			/* Unexpected data in recipient field, disregard.. */
			memset( md, 0, sizeof( struct message_data ) );
			md->type = MSG_UNKNOWN;
			return;
		}
		md->safe = 1;
	}
	else
	{
		if ( irccmp( id_name, chanfix->name ) )
		{
			/* Unexpected data in recipient field, disregard.. */
			memset( md, 0, sizeof( struct message_data ) );
			md->type = MSG_UNKNOWN;
			return;
		}
		md->safe = 0;
	}



	/* Ignore possible colon character */
	if ( *cptr == ':' )
	{
		cptr++;
	}

	/* A CTCP message */
	if ( *cptr == 1 )
	{
		md->type = MSG_CTCP;
		cptr++;
	}

	/* Copy everything from cptr to the end of msg to md->data.
	 * msg is null-terminated, so strncpy should suffice. */
	strncpy( md->data, cptr, IRCLINELEN );

	if ( md->type == MSG_CTCP )
	{
		ctcp_valid = 0;

		/* If CTCP, force termination at next char 001 */
		for ( i = 1; i < strlen( md->data ); i++ )
		{
			if ( md->data[i] == 1 )
			{
				ctcp_valid = 1;
				md->data[i] = 0;
				break;
			}
		}

		if ( !ctcp_valid )
		{
			/* This is a malformed CTCP. Let's just ignore it */
			memset( md, 0, sizeof( struct message_data ) );
			md->type = MSG_UNKNOWN;
			return;
		}
	}
}

/* ----------------------------------------------------------------- */
/*
 * :sabre2th!sabre2th@talking.yoda.is PRIVMSG #carnique :hi C
 */
void get_chanmsg_data(char *msg, struct message_data *md)
{
	/* A channel message? Uninteresting. Let's not bother parsing it */
	memset( md, 0, sizeof( struct message_data ) );
	md->type = MSG_CHANMSG;
	return;
}



/* ----------------------------------------------------------------- */

void process_chanmsg(struct message_data *md)
{
	/* Don't do anything with channel messages. */
}

/* ----------------------------------------------------------------- */

void process_killed(struct message_data *md)
{
	ocf_log(OCF_LOG_DEBUG, "Disconnected. Reconnecting.");
	ocf_log(OCF_LOG_MAIN, "DISC Disconnected. Reconnecting.");
	chanfix = NULL;
	OCF_connect_chanfix(NULL);
	/* We should actually be reconnecting with a delay to prevent
	 * throttling. But calling the delay function, as below, somehow
	 * doesn't work properly.
	 * TODO: fix delay reconnection.
	OCF_connect_chanfix_with_delay(OCF_CONNECT_DELAY);
	*/
}

/* ----------------------------------------------------------------- */

void process_privmsg(struct message_data *md)
{
	struct Client *sender = find_person(md->nick);
	struct ocf_loggedin_user *olu;
	struct command_data cd;
	process_cmd_t funcp;
	int required_flags;

	memset(&cd, 0, sizeof(cd));

	/* Check if the sender is still on the network. */
	if (!sender) {
		/* Send error to owner(s) about non-existing nick */
		ocf_log(OCF_LOG_DEBUG, "pr_pr: nick %s does not exist.", md->nick);
		return;
	}

	/* Only accept commands from opers. */
	if (!IsOper(sender)) {
		/* Send error to owner(s) about non-oper command */
		ocf_log(OCF_LOG_DEBUG, "pr_pr: non-oper %s sent me %s", md->nick, md->data);
		ocf_log(OCF_LOG_PRIVMSG, "PRIV non-oper %s (%s@%s) [%s]: %s",
			md->nick, md->user, md->host, md->server, md->data);
		return;
	}

	ocf_log(OCF_LOG_DEBUG, "Got privmsg from oper %s: %s", md->nick, md->data);
	ocf_log(OCF_LOG_PRIVMSG, "PRIV oper %s (%s@%s) [%s]: %s",
		md->nick, md->user, md->host, md->server, md->data);
	olu = ocf_get_loggedin_user(md->nick);

	/* Check if the user@host of the nickname that sent us this message is the
	 * same as the user@host of the logged in user. If not, log out that nickname`
	 * to avoid hijacking of logged in sessions. */
	if (olu != NULL) {
		if (strcasecmp(md->user, olu->user) != 0 || 
			strcasecmp(md->host, olu->host) != 0 ||
			strcasecmp(md->server, olu->server) != 0) {
			ocf_log(OCF_LOG_MAIN, 
				"WARN Possible session hijack for %s: logged in with %s@%s [%s], now %s@%s [%s].",
				md->nick, olu->user, olu->host, olu->server, md->user, md->host, md->server);
			send_notice_to_loggedin_except(olu, OCF_NOTICE_FLAGS_MAIN, 
				"Possible session hijack for %s: logged in with %s@%s [%s], now %s@%s [%s].",
				md->nick, olu->user, olu->host, olu->server, md->user, md->host, md->server);
			ocf_user_logout(md->nick);
			olu = NULL;
		}
	}

	cd.loggedin_user_p = olu;
	get_command_data(md->data, &cd);

	/* Check whether the user is logged in. */
	if (olu == NULL && command_requires_loggedin(cd.command)) {
		chanfix_send_privmsg(md->nick, "You need to login to use this command.");
		return;
	}

	/* Check whether the user has the necessary flags. */
	required_flags = cmd_flags_required[cd.command];
	if (olu && required_flags && !ocf_user_has_flag(olu->ocf_user_p, required_flags)) {
		chanfix_send_privmsg(md->nick, "You don't have the %c flag.",
			ocf_user_flag_get_char(required_flags));
		return;
	}

	/* Check whether the syntax of the command is correct. */
	if (cd.wrong_syntax) {
		ocf_log(OCF_LOG_DEBUG, "Wrong syntax.");
		process_cmd_wrong_syntax(md, &cd);
		return;
	}

	/* Get the function pointer to the function to process this command. */
	funcp = process_cmd_functions[cd.command];
	if (funcp == NULL) {
		ocf_log(OCF_LOG_DEBUG, "Eeks! NULL function pointer in process_privmsg!");
		chanfix_send_privmsg(md->nick,
			"ERROR - NULL function pointer in process_privmsg. Please report to Garion.");
		return;
	}
	funcp(md, &cd);
}

/* ----------------------------------------------------------------- */

void send_command_list(char* nick) {
	char cmds[IRCLINELEN];
	int i;
	struct ocf_loggedin_user *olu;
	cmds[0] = 0;
	olu = ocf_get_loggedin_user(nick);

	for (i = 1; i <= NUM_COMMANDS; i++) {
		/* Only send help for commands that require loggedin if the 
		 * user is actually logged in. */
		if (command_requires_loggedin(i)) {
			if (olu) {
				strcat(cmds, cmd_word[i]);
				if (i != NUM_COMMANDS) {
					strcat(cmds, " ");
				}
			}
		} else {
			strcat(cmds, cmd_word[i]);
			if (i != NUM_COMMANDS) {
				strcat(cmds, " ");
			}
		}
	}
	chanfix_send_privmsg(nick, "List of commands:");
	chanfix_send_privmsg(nick, "%s", cmds);
}

/* ----------------------------------------------------------------- */

void process_cmd_wrong_syntax(struct message_data *md, struct command_data *cd)
{
	int parc;
	char *parv[MAXPARA];
	char help[IRCLINELEN];

	get_command_help(cd->command, help);
	parc = tokenize_string_separator(help, parv, '\n');

	if (parc == 0) {
		chanfix_send_privmsg(md->nick, 
			"I do not know the syntax for this command.");
	} else {
		chanfix_send_privmsg(md->nick, "Please use the following syntax:");
		chanfix_send_privmsg(md->nick, "%s", parv[0]);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_unknown(struct message_data *md, struct command_data *cd)
{
	chanfix_send_privmsg(md->nick, 
		"Unknown command. Please use HELP for help.");
}

/* ----------------------------------------------------------------- */
/* HELP [command] 
 * cd->params - command
 */

void process_cmd_help(struct message_data *md, struct command_data *cd)
{
	int parc;
	char *parv[MAXPARA];
	char help[IRCLINELEN];
	int i, cmd, flag;
	struct ocf_loggedin_user *olu;
	olu = ocf_get_loggedin_user(md->nick);

	if (cd->params[0]) {
		cmd = get_command_int_from_word(cd->params);
		/* Don't show help for commands that require you to be logged in,
		 * unless you're actually logged in. */
		if ( cmd == CMD_UNKNOWN ||
			(command_requires_loggedin(cmd) && olu == NULL) ) {
			chanfix_send_privmsg(md->nick, "Unknown command %s.", cd->params);
		} else {
			get_command_help(cmd, help);
			parc = tokenize_string_separator(help, parv, '\n');
			for (i = 0; i < parc; i++) {
				chanfix_send_privmsg(md->nick, "%s", parv[i]);
			}
			flag = cmd_flags_required[cmd];
			if (flag) {
				chanfix_send_privmsg(md->nick, "This command requires flag %c.", 
					ocf_user_flag_get_char(flag));
			}
		}
	} else {
		/* TODO: don't show commands you can't execute. */
		send_command_list(md->nick);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_score(struct message_data *md, struct command_data *cd)
{
	struct Channel* chptr;
	int dummy;

	/* Check if there are scores in the database for this channel. */
	DB_set_read_channel(cd->channel);
	if (!DB_get_op_scores( &dummy, 1 )) {
		chanfix_send_privmsg(md->nick, 
			"There are no scores in the database for \"%s\".", cd->channel);
		return;
	}
	
	if (cd->nick[0] != 0) {
		process_cmd_score_nick(md, cd, 0);
		return;
	}

	if (cd->user[0] != 0) {
		process_cmd_score_hostmask(md, cd, 0);
		return;
	}

	chptr = hash_find_channel(cd->channel);

	if (chptr) {
		process_cmd_score_existing_channel(md, cd, chptr, 0);
	} else {
		process_cmd_score_nonexisting_channel(md, cd, 0);
	}

}

/* ----------------------------------------------------------------- */

void process_cmd_cscore(struct message_data *md, struct command_data *cd)
{
	struct Channel* chptr;
	int dummy;

	/* Check if there are scores in the database for this channel. */
	DB_set_read_channel(cd->channel);
	if (!DB_get_op_scores( &dummy, 1 )) {
		chanfix_send_privmsg(md->nick, 
			"~! %s", cd->channel);
		return;
	}
	
	if (cd->nick[0] != 0) {
		process_cmd_score_nick(md, cd, 1);
		return;
	}

	if (cd->user[0] != 0) {
		process_cmd_score_hostmask(md, cd, 1);
		return;
	}

	chptr = hash_find_channel(cd->channel);

	if (chptr) {
		process_cmd_score_existing_channel(md, cd, chptr, 1);
	} else {
		process_cmd_score_nonexisting_channel(md, cd, 1);
	}

}

/* ----------------------------------------------------------------- */
/* TODO: this function is too long. Break it up. */

void process_cmd_score_existing_channel(struct message_data *md, 
										struct command_data *cd, 
										struct Channel* chptr,
										int compact)
{
	struct OCF_topops channelscores;
	struct Client *clptr;
	struct Membership* mship;
	dlink_node *node_client;
	int i, status;
	char tmpBuf[IRCLINELEN];
	char tmpBuf2[IRCLINELEN];
	int num_opped = 0,
		num_peons = 0,
		num_clients = 0,
		numtopscores;
	char **oppedusers, **oppedhosts, **noppedusers, **noppedhosts;

	ocf_log(OCF_LOG_SCORE, 
		"SCOR %s requested scores for channel \"%s\"", 
		md->nick, cd->channel);

	numtopscores = settings_get_int("num_top_scores");
	if (numtopscores < 1 || numtopscores > OPCOUNT) {
		numtopscores = OCF_NUM_TOP_SCORES;
	}

	/* Fill the (n)opped user stuff */
	num_clients = dlink_list_length(&(chptr->members));
	oppedusers = MyMalloc(num_clients * sizeof(char*));
	oppedhosts = MyMalloc(num_clients * sizeof(char*));
	noppedusers = MyMalloc(num_clients * sizeof(char*));
	noppedhosts = MyMalloc(num_clients * sizeof(char*));

	/* Create an array with unique userhosts of the opped, and the
	 * peon clients in the channel */
	DLINK_FOREACH(node_client, chptr->members.head) {
		mship = (struct Membership*)(node_client->data);
		clptr = mship->client_p;
		status = OCF_determine_client_status(mship);

		if (status & OCF_OPPED) {
			num_opped = add_userhost_to_array(oppedusers, oppedhosts, num_opped, 
				clptr->username, clptr->host);
		} else {
			num_peons = add_userhost_to_array(noppedusers, noppedhosts, num_peons, 
				clptr->username, clptr->host);
		}
	}

	memset(&channelscores, 0, sizeof(channelscores));
	DB_set_read_channel(chptr->chname);
	DB_poll_channel( &channelscores, 
		oppedusers, oppedhosts, num_opped, 
		noppedusers, noppedhosts, num_peons);

	/* (n)opped users no longer needed */
	MyFree(oppedusers);
	MyFree(oppedhosts);
	MyFree(noppedusers);
	MyFree(noppedhosts);

	if (channelscores.topscores[0] == 0) {
		if (compact) {
			chanfix_send_privmsg(md->nick, "~! %s", cd->channel);
		} else {
			chanfix_send_privmsg(md->nick, 
				"There are no scores in the database for \"%s\".",
				cd->channel);
		}
		return;
	}

	/* Top scores */
	if (!compact) {
		chanfix_send_privmsg(md->nick, 
			"Top %d scores for channel \"%s\" in the database:",
			numtopscores, cd->channel);
	}
	tmpBuf[0] = 0;
	i = 0;
	if (compact) {
		sprintf(tmpBuf, "~S %s ", cd->channel);
	}
	while (i < numtopscores && channelscores.topscores[i] > 0) {
		if (compact) {
			sprintf(tmpBuf2, "%d ", channelscores.topscores[i]);
		} else {
			sprintf(tmpBuf2, "%d, ", channelscores.topscores[i]);
		}
		strcat(tmpBuf, tmpBuf2);
		i++;
	}
	if (!compact) {
		tmpBuf[strlen(tmpBuf) - 2] = '.';
	}
	chanfix_send_privmsg(md->nick, "%s", tmpBuf);

	/* Top scores of opped clients */
	if (!compact) {
		chanfix_send_privmsg(md->nick, 
			"Top %d scores for ops in channel \"%s\" in the database:",
			numtopscores, chptr->chname);
	}

	tmpBuf[0] = 0;
	i = 0;
	if (compact) {
		sprintf(tmpBuf, "~O %s ", cd->channel);
	}
	while (i < numtopscores && channelscores.currentopscores[i] > 0) {
		if (compact) {
			sprintf(tmpBuf2, "%d ", channelscores.currentopscores[i]);
		} else {
			sprintf(tmpBuf2, "%d, ", channelscores.currentopscores[i]);
		}
		strcat(tmpBuf, tmpBuf2);
		i++;
	}
	if (strlen(tmpBuf) > 0) {
		if (!compact) {
			tmpBuf[strlen(tmpBuf) - 2] = '.';
		}
		chanfix_send_privmsg(md->nick, "%s", tmpBuf);
	} else {
		chanfix_send_privmsg(md->nick, "None.");
	}

	/* Top scores of non-opped clients */
	if (!compact) {
		chanfix_send_privmsg(md->nick, 
			"Top %d scores for non-ops in channel \"%s\" in the database:",
			numtopscores, chptr->chname);
	}

	tmpBuf[0] = 0;
	i = 0;
	if (compact) {
		sprintf(tmpBuf, "~N %s ", cd->channel);
	}
	while (i < numtopscores && channelscores.currentnopscores[i] > 0) {
		if (compact) {
			sprintf(tmpBuf2, "%d ", channelscores.currentnopscores[i]);
		} else {
			sprintf(tmpBuf2, "%d, ", channelscores.currentnopscores[i]);
		}
		strcat(tmpBuf, tmpBuf2);
		i++;
	}
	if (strlen(tmpBuf) > 0) {
		if (!compact) {
			tmpBuf[strlen(tmpBuf) - 2] = '.';
		}
		chanfix_send_privmsg(md->nick, "%s", tmpBuf);
	} else {
		chanfix_send_privmsg(md->nick, "None.");
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_score_nonexisting_channel(struct message_data *md, 
										   struct command_data *cd,
										   int compact)
{
	struct OCF_topops channelscores;
	char tmpBuf[IRCLINELEN];
	char tmpBuf2[IRCLINELEN];
	int i;
	int numtopscores = settings_get_int("num_top_scores");
	if (numtopscores < 1 || numtopscores > OPCOUNT) {
		numtopscores = OCF_NUM_TOP_SCORES;
	}

	ocf_log(OCF_LOG_SCORE, 
		"SCOR %s requested scores for channel \"%s\"", 
		md->nick, cd->channel);

	memset(&channelscores, 0, sizeof(channelscores));
	DB_set_read_channel(cd->channel);
	DB_poll_channel( &channelscores, (char**)1, (char**)1, 0, (char**)1, (char**)1, 0);
	
	if (channelscores.topscores[0] == 0) {
		if (compact) {
			chanfix_send_privmsg(md->nick, "~! %s", cd->channel);
		} else {
			chanfix_send_privmsg(md->nick, 
				"There are no scores in the database for \"%s\".",
				cd->channel);
		}
		return;
	}

	/* Top scores */
	if (!compact) {
		chanfix_send_privmsg(md->nick, 
			"Top %d scores for channel \"%s\" in the database:",
			numtopscores, cd->channel);
	}
	tmpBuf[0] = 0;
	i = 0;
	if (compact) {
		sprintf(tmpBuf, "~S %s ", cd->channel);
	}
	while (i < numtopscores && channelscores.topscores[i] > 0) {
		if (compact) {
			sprintf(tmpBuf2, "%d ", channelscores.topscores[i]);
		} else {
			sprintf(tmpBuf2, "%d, ", channelscores.topscores[i]);
		}
		strcat(tmpBuf, tmpBuf2);
		i++;
	}
	chanfix_send_privmsg(md->nick, "%s", tmpBuf);
	if (compact) {
		chanfix_send_privmsg(md->nick, "~O %s", cd->channel);
		chanfix_send_privmsg(md->nick, "~N %s", cd->channel);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_score_nick(struct message_data *md, struct command_data *cd,
							int compact)
{
	struct Client *clptr;
	struct OCF_chanuser highscores;
	char* users[1];
	char* hosts[1];

	if (!settings_get_bool("enable_cmd_score_nick")) {
		chanfix_send_privmsg(md->nick, "This command has been disabled.");
		return;
	}

	clptr = find_client(cd->nick);

	if (!clptr) {
		if (compact) {
			chanfix_send_privmsg(md->nick, "~U %s no@such.nick 0", cd->channel);
		} else {
			chanfix_send_privmsg(md->nick, "No such nick %s.", cd->nick);
		}
		return;
	}
	
	users[0] = clptr->username;
	hosts[0] = clptr->host;	

	ocf_log(OCF_LOG_SCORE, 
		"SCOR %s requested score for %s (%s@%s) in channel \"%s\"", 
		md->nick, cd->nick, clptr->username, clptr->host, cd->channel);

	highscores.score = 0;
	DB_set_read_channel(cd->channel);
	DB_get_top_user_hosts(&highscores, users, hosts, 1, 1);

	if (compact) {
		chanfix_send_privmsg(md->nick, "~U %s %s@%s %d",
			cd->channel, clptr->username, clptr->host, highscores.score);
	} else {
		chanfix_send_privmsg(md->nick, 
			"Score for \"%s@%s\" in channel \"%s\": %d.",
			clptr->username, clptr->host, cd->channel, highscores.score);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_score_hostmask(struct message_data *md, struct command_data *cd,
								int compact)
{
	struct OCF_chanuser highscores;
	char* users[1];
	char* hosts[1];

	if (!settings_get_bool("enable_cmd_score_userhost")) {
		chanfix_send_privmsg(md->nick, "This command has been disabled.");
		return;
	}

	users[0] = cd->user;
	hosts[0] = cd->host;	

	ocf_log(OCF_LOG_SCORE, 
		"SCOR %s requested score for %s@%s in channel \"%s\"", 
		md->nick, cd->user, cd->host, cd->channel);

	highscores.score = 0;
	DB_set_read_channel(cd->channel);
	DB_get_top_user_hosts(&highscores, users, hosts, 1, 1);

	if (compact) {
		chanfix_send_privmsg(md->nick, "~U %s %s@%s %d",
			cd->channel, cd->user, cd->host, highscores.score);
	} else {
		chanfix_send_privmsg(md->nick, 
			"Score for \"%s@%s\" in channel \"%s\": %d.",
			cd->user, cd->host, cd->channel, highscores.score);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_chanfix(struct message_data *md, struct command_data *cd)
{
	struct Channel* chptr;
	char tmpBuf[IRCLINELEN];
	unsigned int flags;

	/* Check whether manual chanfix has been disabled in the settings. */
	if (!settings_get_bool("enable_chanfix")) {
		chanfix_send_privmsg(md->nick, 
			"Sorry, manual chanfixes are currently disabled.");
		send_notice_to_loggedin_except(cd->loggedin_user_p, 
			OCF_NOTICE_FLAGS_CHANFIX,
			"[currently disabled] %s requested manual fix for %s", 
			cd->loggedin_user_p->nick, cd->channel);
		ocf_log(OCF_LOG_CHANFIX, 
			"CHFX [currently disabled] %s requested manual fix for %s", 
			cd->loggedin_user_p->nick, cd->channel);
		return;
	}

	/* Check whether enough servers are currently linked. */
	autofix_disabled_due_to_split = !check_minimum_servers_linked();

	/* If not, bail out. */
	if (autofix_disabled_due_to_split) {
		chanfix_send_privmsg(md->nick, 
			"Sorry, chanfix is currently disabled because not enough servers are linked.");
		send_notice_to_loggedin_except(cd->loggedin_user_p, 
			OCF_NOTICE_FLAGS_CHANFIX,
			"[not enough servers] %s requested manual fix for %s", 
			cd->loggedin_user_p->nick, cd->channel);
		ocf_log(OCF_LOG_CHANFIX, 
			"CHFX [not enough servers] %s requested manual fix for %s", 
			cd->loggedin_user_p->nick, cd->channel);
		return;
	}

	/* Only allow chanfixes for channels that are in the database. */
	if (!DB_channel_exists(cd->channel)) {
		chanfix_send_privmsg(md->nick, 
			"There are no scores in the database for \"%s\".", cd->channel);
		return;
	}

	/* Don't fix a channel being chanfixed. */
	if (is_being_chanfixed(cd->channel)) {
		chanfix_send_privmsg(md->nick, 
			"Channel \"%s\" is already being manually fixed.", cd->channel);
		return;
	}

	/* Don't fix a channel being autofixed without OVERRIDE flag. */
	if (is_being_autofixed(cd->channel)) {
		if (cd->flags == 0) {
			chanfix_send_privmsg(md->nick, 
				"Channel \"%s\" is being automatically fixed. Append the OVERRIDE flag to force a manual fix.", 
				cd->channel);
			return;
		} else {
			/* We're going to manually fix this instead of autofixing it,
			 * so remove this channel from the autofix list. */
			del_autofix_channel(cd->channel);
		}
	}

	/* Don't fix a blocked channel. */
	DB_set_read_channel(cd->channel);
	flags = DB_channel_get_flag();
	if (flags & OCF_CHANNEL_FLAG_BLOCK) {
		chanfix_send_privmsg(md->nick, 
			"Channel \"%s\" is BLOCKED.", cd->channel);
		return;
	}

	/* Don't fix an alerted channel without the OVERRIDE flag. */
	DB_set_read_channel(cd->channel);
	flags = DB_channel_get_flag();
	if (flags & OCF_CHANNEL_FLAG_ALERT && cd->flags == 0) {
		chanfix_send_privmsg(md->nick, 
			"Alert: channel \"%s\" has notes. Use \"INFO %s\" to read them. Append the OVERRIDE flag to force a manual fix.", 
			cd->channel, cd->channel);
		return;
	}

	/* Only allow chanfixes for existing channels. */
	chptr = hash_find_channel(cd->channel);
	if (!chptr) {
		chanfix_send_privmsg(md->nick, "No such channel %s.", cd->channel);
		return;
	}

	/* Fix the channel */
	add_chanfix_channel(cd->channel);

	/* Add note to the channel about this manual fix */
	snprintf(tmpBuf, sizeof(tmpBuf), "CHANFIX by %s", 
		cd->loggedin_user_p->ocf_user_p->name);
	DB_set_write_channel(cd->channel);
	DB_channel_add_note(cd->loggedin_user_p->ocf_user_p->id, 
		time(NULL), tmpBuf);

	/* Log the chanfix */
	chanfix_send_privmsg(md->nick, 
		"Manual chanfix acknowledged for %s", cd->channel);
	send_notice_to_loggedin_except(cd->loggedin_user_p, OCF_NOTICE_FLAGS_CHANFIX,
		"%s requested manual fix for %s", 
		cd->loggedin_user_p->nick, cd->channel);
	ocf_log(OCF_LOG_CHANFIX, "CHFX %s requested manual fix for %s", 
		cd->loggedin_user_p->nick, cd->channel);
}

/* ----------------------------------------------------------------- */
/* Tries to login the user sending the login command. */

void process_cmd_login(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;
	struct ocf_loggedin_user *olu;
	char hostmask[USERLEN + HOSTLEN + 1];

	/* Only accept the LOGIN command if it has been sent to user@server,
	 * and not if it's been sent to the chanfix nick. */
	if (!md->safe) {
		chanfix_send_privmsg(md->nick, 
			"For security reasons, you must message your login to %s@%s instead of %s.",
			chanfix->username, me.name, chanfix->name);
		return;
	}

	/* Get the user belonging to this username. */
	ou = ocf_get_user(cd->name);
	if (!ou) {
		/* This user doesn't exist. */
		chanfix_send_privmsg(md->nick, "Incorrect username or password.");
		ocf_log(OCF_LOG_USERS, 
			"LOGI Login attempt for non-existing user %s by %s (%s@%s) [%s].", 
			cd->name, md->nick, md->user, md->host, md->server);
		return;
	} else if (ou->deleted) {
		/* This user has been deleted. */
		chanfix_send_privmsg(md->nick, "Incorrect username or password.");
		ocf_log(OCF_LOG_USERS, 
			"LOGI Login attempt for deleted user %s by %s (%s@%s) [%s].", 
			ou->name, md->nick, md->user, md->host, md->server);
		return;
	}

	/* Check if a user with this nick is already logged in.
	 * If so, log them out first. This prevents users getting the wrong
	 * rights for any reason. */
	olu = ocf_get_loggedin_user(md->nick);
	if (olu) {
		ocf_user_logout(md->nick);
	}

	/* Check hostmask of user. */
	snprintf(hostmask, sizeof(hostmask), "%s@%s", md->user, md->host);
	if (!ocf_user_has_hostmask(ou, hostmask)) {
		chanfix_send_privmsg(md->nick, "Incorrect username or password.");
		send_notice_to_loggedin(OCF_NOTICE_FLAGS_LOGINOUT,
			"Login failed (wrong hostmask) as %s by %s (%s@%s).", 
			ou->name, md->nick, md->user, md->host);
		ocf_log(OCF_LOG_USERS, "LOGI Login failed (wrong hostmask) as %s by %s (%s@%s) [%s].", 
			ou->name, md->nick, md->user, md->host, md->server);
		return;
	}

	/* Hostmask ok, now the password check. */
	if (ocf_user_check_password(ou, cd->password)) {
		ocf_log(OCF_LOG_DEBUG, "Correct password for %s", cd->name);
		olu = ocf_user_login(md->nick, md->user, md->host, md->server, ou);
		chanfix_send_privmsg(md->nick,
			"Authentication successful. Welcome, %s.", md->nick);
		if (olu->ocf_user_p->flags) {
			chanfix_send_privmsg(md->nick, "Your flags are %s.", 
				ocf_user_flags_to_string(olu->ocf_user_p->flags));
		}
		if (olu->ocf_user_p->noticeflags) {
			chanfix_send_privmsg(md->nick, "Your umodes are %s.", 
				ocf_user_notice_flags_to_string(olu->ocf_user_p->noticeflags));
		}
		send_notice_to_loggedin_except(olu, OCF_NOTICE_FLAGS_LOGINOUT,
			"Login as %s by %s (%s@%s) [%s].", olu->ocf_user_p->name,
			md->nick, md->user, md->host, md->server);
		ocf_log(OCF_LOG_USERS, "LOGI %s has logged in as %s from %s@%s [%s].",
			md->nick, ou->name, md->user, md->host, md->server);
	} else {
		/* Password was wrong. Log failure. */
		chanfix_send_privmsg(md->nick, "Incorrect username or password.");
		send_notice_to_loggedin(OCF_NOTICE_FLAGS_LOGINOUT,
			"Login failed (wrong password) as %s by %s (%s@%s) [%s].", 
			ou->name, md->nick, md->user, md->host, md->server);
		ocf_log(OCF_LOG_USERS, "LOGI Login failed (wrong password) as %s by %s (%s@%s) [%s].", 
			ou->name, md->nick, md->user, md->host, md->server);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_logout(struct message_data *md, struct command_data *cd)
{
	ocf_user_logout(md->nick);
	chanfix_send_privmsg(md->nick, "You have logged out.");
	send_notice_to_loggedin(OCF_NOTICE_FLAGS_LOGINOUT,
		"%s has logged out.", md->nick);
	ocf_log(OCF_LOG_USERS, "LOGO %s has logged out.", md->nick);
}

/* ----------------------------------------------------------------- */

void process_cmd_passwd(struct message_data *md, struct command_data *cd)
{
	/* Test if old password is correct. */
	if (!ocf_user_check_password(cd->loggedin_user_p->ocf_user_p, cd->params)) {
		chanfix_send_privmsg(md->nick, "Old password does not match.");
		return;
	}
	
	ocf_user_set_password(cd->loggedin_user_p->ocf_user_p, cd->password);
	ocf_users_updated();
	chanfix_send_privmsg(md->nick, "Your password has been changed.");
}

/* ----------------------------------------------------------------- */

void process_cmd_history(struct message_data *md, struct command_data *cd)
{
	int i = 0;
	struct OCF_topops result;

	DB_set_read_channel(cd->channel);
	DB_poll_channel(&result, (char**)1, (char**)1, 0, (char**)1, (char**)1, 0);

	if (result.manualfixhistory[0] == 0) {
		chanfix_send_privmsg(md->nick, 
			"Channel \"%s\" has never been manually fixed.", cd->channel);
		return;
	}

	chanfix_send_privmsg(md->nick, 
			"Channel \"%s\" has been manually fixed on:", cd->channel);
	while (result.manualfixhistory[i] > 0) {
		chanfix_send_privmsg(md->nick, "%s", 
			seconds_to_datestring((time_t)result.manualfixhistory[i]));
		i++;
	}
	chanfix_send_privmsg(md->nick, "End of list.");
}

/* ----------------------------------------------------------------- */

void process_cmd_info(struct message_data *md, struct command_data *cd)
{
	int num_notes, i;
	unsigned int flags;
	struct OCF_channote* note_p;

	if (!DB_channel_exists(cd->channel)) {
		chanfix_send_privmsg(md->nick, 
			"No information on %s in the database.", cd->channel);
		return;
	}

	chanfix_send_privmsg(md->nick, "Information on \"%s\":", cd->channel);

	DB_set_read_channel(cd->channel);
	flags = DB_channel_get_flag();
	if (flags & OCF_CHANNEL_FLAG_BLOCK) {
		chanfix_send_privmsg(md->nick, "%s is BLOCKED.", cd->channel);
	} else if (flags & OCF_CHANNEL_FLAG_ALERT) {
		chanfix_send_privmsg(md->nick, "%s is ALERTED.", cd->channel);
	}

	if (is_being_chanfixed(cd->channel)) {
		chanfix_send_privmsg(md->nick, "%s is being chanfixed.", cd->channel);
	}
	if (is_being_autofixed(cd->channel)) {
		chanfix_send_privmsg(md->nick, "%s is being autofixed.", cd->channel);
	}

	num_notes = DB_channel_note_count();
	if (num_notes > 0) {
		chanfix_send_privmsg(md->nick, "Notes (%d):", num_notes);
		for (i = 1; i <= num_notes; i++) {
			note_p = DB_channel_get_note(i);
			if (note_p) {
				if (ocf_user_get_name_from_id(note_p->id)) {
					chanfix_send_privmsg(md->nick, "[%d:%s] %s %s", 
						i, ocf_user_get_name_from_id(note_p->id), 
						seconds_to_datestring((time_t)note_p->timestamp), 
						note_p->note);
				} else {
					chanfix_send_privmsg(md->nick, "[%d:UNKNOWN] %s %s", 
						i, seconds_to_datestring((time_t)note_p->timestamp), 
						note_p->note);
				}
			}
		}
	}
	chanfix_send_privmsg(md->nick, "End of information.");

	ocf_log(OCF_LOG_CHANFIX,
		"INFO %s requested info` for \"%s\"",
		md->nick, cd->channel);
}

/* ----------------------------------------------------------------- */

void process_cmd_oplist(struct message_data *md, struct command_data *cd)
{
	struct OCF_chanuser oplist[10];
	int scorecount[1];
	int r, i;

	DB_set_read_channel(cd->channel);

	if ( !DB_get_op_scores( scorecount, 1 ) )
	{
		chanfix_send_privmsg(md->nick,
			"There are no scores in the database for \"%s\".",
			cd->channel);
		return;
    }

	ocf_log(OCF_LOG_CHANFIX,
		"OPLI %s requested oplist for \"%s\"",
		md->nick, cd->channel);

	send_notice_to_loggedin(OCF_NOTICE_FLAGS_MAIN, 
		"%s has requested OPLIST for \"%s\"", 
		md->nick, cd->channel);

	r = DB_get_oplist( oplist, 10 );

	if ( r == 10 )
	{
		chanfix_send_privmsg(md->nick,
			"Top 10 unique op hostmasks in channel \"%s\":",
			cd->channel);
	}
	else
	{
		chanfix_send_privmsg(md->nick,
			"Found %d unique op hostmasks in channel \"%s\":",
			r, cd->channel);
	}

	for( i = 0; i < r; i++ )
	{
		chanfix_send_privmsg(md->nick,
			"%2d. %4d %s@%s",
			( i + 1 ), oplist[i].score, oplist[i].user, oplist[i].host);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_addnote(struct message_data *md, struct command_data *cd)
{
	int num_notes;

	DB_set_write_channel(cd->channel);
	num_notes = DB_channel_add_note(cd->loggedin_user_p->ocf_user_p->id, time(NULL), cd->params);

	chanfix_send_privmsg(md->nick, "Added note to %s (%d notes).",
		cd->channel, num_notes);
	send_notice_to_loggedin_except(cd->loggedin_user_p, OCF_NOTICE_FLAGS_NOTES,
		"%s added note to %s (%d notes).", cd->loggedin_user_p->ocf_user_p->name,
		cd->channel, num_notes);
	ocf_log(OCF_LOG_NOTES, "NOTE %s added note to %s (%d notes).", 
		cd->loggedin_user_p->ocf_user_p->name, cd->channel, num_notes);
}

/* ----------------------------------------------------------------- */

void process_cmd_delnote(struct message_data *md, struct command_data *cd)
{
	struct OCF_channote* note_p;
	int res;

	if (!DB_channel_exists(cd->channel)) {
		chanfix_send_privmsg(md->nick, 
			"No information on %s in the databas.", cd->channel);
		return;
	}
	
	DB_set_read_channel(cd->channel);
	note_p = DB_channel_get_note(cd->id);

	if (note_p == NULL) {
		chanfix_send_privmsg(md->nick, 
			"No such note %d for channel %s", cd->id, cd->channel);
		return;
	}

	DB_set_write_channel(cd->channel);
	res = DB_channel_del_note(cd->id);

	if (res) {
		chanfix_send_privmsg(md->nick, 
			"Note %d for channel %s deleted.", cd->id, cd->channel);
		send_notice_to_loggedin_except(cd->loggedin_user_p, OCF_NOTICE_FLAGS_NOTES,
			"%s deleted note %d for channel %s.", 
			cd->loggedin_user_p->ocf_user_p->name, cd->id, cd->channel);
		ocf_log(OCF_LOG_NOTES, "NOTE %s deleted note %d for channel %s.", 
			cd->loggedin_user_p->ocf_user_p->name, cd->id, cd->channel);
	} else {
		chanfix_send_privmsg(md->nick, 
			"Error when deleting note %d for channel %s", cd->id, cd->channel);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_whois(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;

	ou = ocf_get_user(cd->name);
	if (ou) {
		chanfix_send_privmsg(md->nick, "User: %s", ou->name);
		if (ou->flags == 0) {
			chanfix_send_privmsg(md->nick, "User %s has no flags.", ou->name);
		} else {
			chanfix_send_privmsg(md->nick, "User %s has flags: %s", 
				ou->name, ocf_user_flags_to_string(ou->flags));
		}
		chanfix_send_privmsg(md->nick, "Hosts: %s", ou->hostmasks);
	} else {
		chanfix_send_privmsg(md->nick, "No such user %s.", cd->name);
	}
}

/* ----------------------------------------------------------------- */
/* Add a flag to a user.
 * cd->name = user to be modified.
 * cd->flags = single flag to be added.
 */
void process_cmd_addflag(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;

	ou = ocf_get_user(cd->name);

	if (ou == NULL){
		chanfix_send_privmsg(md->nick, "No such user %s.", cd->name);
		return;
	}

	if (cd->flags == OCF_USER_FLAGS_OWNER) {
		chanfix_send_privmsg(md->nick, "You cannot add an owner flag.");
		return;
	}

	if (cd->flags == OCF_USER_FLAGS_USERMANAGER &&
		!ocf_user_has_flag(cd->loggedin_user_p->ocf_user_p, OCF_USER_FLAGS_OWNER)) {
		chanfix_send_privmsg(md->nick, "Only an owner can add the user management flag.");
		return;
	}

	if (ocf_user_has_flag(ou, cd->flags)) {
		chanfix_send_privmsg(md->nick, "User %s already has flag %c.",
			ou->name, ocf_user_flag_get_char(cd->flags));
		return;
	}

	ocf_user_add_flags(ou, cd->flags);
	ocf_users_updated();
	chanfix_send_privmsg(md->nick, "Added flag %c to user %s.",
		ocf_user_flag_get_char(cd->flags), ou->name);
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_USER,
		"%s added flag %c to user %s.", cd->loggedin_user_p->nick, 
		ocf_user_flag_get_char(cd->flags), ou->name);
	send_notice_to_loggedin_username(ou->name, "%s gave you the %c flag.",
		cd->loggedin_user_p->nick, ocf_user_flag_get_char(cd->flags));
	ocf_log(OCF_LOG_USERS, "USFL %s added flag %c to user %s.", 
		cd->loggedin_user_p->ocf_user_p->name, 
		ocf_user_flag_get_char(cd->flags), ou->name);
}

/* ----------------------------------------------------------------- */

void process_cmd_delflag(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;
	ou = ocf_get_user(cd->name);

	if (ou == NULL){
		chanfix_send_privmsg(md->nick, "No such user %s.", cd->name);
		return;
	}

	if (cd->flags == OCF_USER_FLAGS_OWNER) {
		chanfix_send_privmsg(md->nick, "You cannot delete an owner flag.");
		return;
	}

	if (cd->flags == OCF_USER_FLAGS_USERMANAGER &&
		!ocf_user_has_flag(cd->loggedin_user_p->ocf_user_p, OCF_USER_FLAGS_OWNER)) {
		chanfix_send_privmsg(md->nick, "Only an owner can delete the user management flag.");
		return;
	}

	if (!ocf_user_has_flag(ou, cd->flags)) {
		chanfix_send_privmsg(md->nick, "User %s does not have flag %c.",
			ou->name, ocf_user_flag_get_char(cd->flags));
		return;
	}

	ocf_user_del_flags(ou, cd->flags);
	ocf_users_updated();
	chanfix_send_privmsg(md->nick, "Deleted flag %c of user %s.",
		ocf_user_flag_get_char(cd->flags), ou->name);
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_USER,
		"%s deleted flag %c from user %s.", cd->loggedin_user_p->nick, 
		ocf_user_flag_get_char(cd->flags), ou->name);
	send_notice_to_loggedin_username(ou->name, "%s deleted your %c flag.",
		cd->loggedin_user_p->nick, ocf_user_flag_get_char(cd->flags));
	ocf_log(OCF_LOG_USERS, "USFL %s deleted flag %c from user %s.", 
		cd->loggedin_user_p->ocf_user_p->name,
		ocf_user_flag_get_char(cd->flags), ou->name);
}

/* ----------------------------------------------------------------- */

void process_cmd_addhost(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;
	int result;
	ou = ocf_get_user(cd->name);

	if (ou == NULL){
		chanfix_send_privmsg(md->nick, "No such user %s.", cd->name);
		return;
	}

	if (ocf_user_has_hostmask(ou, cd->hostmask)) {
		chanfix_send_privmsg(md->nick, "User %s already has hostmask %s.", 
			cd->name, cd->hostmask);
		return;
	}

	result = ocf_user_add_hostmask(ou, cd->hostmask);
	if (!result) {
		chanfix_send_privmsg(md->nick, "Failed adding hostmask %s to user %s.",
			cd->hostmask, ou->name);
		return;
	}

	ocf_users_updated();
	chanfix_send_privmsg(md->nick, "Added hostmask %s to user %s.",
		cd->hostmask, ou->name);
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_USER,
		"%s added hostmask %s to user %s.", cd->loggedin_user_p->nick, 
		cd->hostmask, ou->name);
	send_notice_to_loggedin_username(ou->name, "%s added you the hostmask %s",
		cd->loggedin_user_p->nick, cd->hostmask);
	ocf_log(OCF_LOG_USERS, "USHM %s added hostmask %s to user %s.", cd->loggedin_user_p->nick, 
		cd->hostmask, ou->name);
}

/* ----------------------------------------------------------------- */

void process_cmd_delhost(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;
	ou = ocf_get_user(cd->name);

	if (ou == NULL){
		chanfix_send_privmsg(md->nick, "No such user %s.", cd->name);
		return;
	}

	if (!ocf_user_has_hostmask(ou, cd->hostmask)) {
		chanfix_send_privmsg(md->nick, "User %s doesn't have hostmask %s.", 
			cd->name, cd->hostmask);
		return;
	}

	ocf_user_del_hostmask(ou, cd->hostmask);
	ocf_users_updated();
	chanfix_send_privmsg(md->nick, "Deleted hostmask %s from user %s.",
		cd->hostmask, ou->name);
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_USER,
		"%s deleted hostmask %s from user %s.", cd->loggedin_user_p->nick, 
		cd->hostmask, ou->name);
	send_notice_to_loggedin_username(ou->name, "%s deleted your hostmask %s",
		cd->loggedin_user_p->nick, cd->hostmask);
	ocf_log(OCF_LOG_USERS, "USHM %s deleted hostmask %s from user %s.", cd->loggedin_user_p->nick, 
		cd->hostmask, ou->name);
}

/* ----------------------------------------------------------------- */
/* Add a user.
 * cd->name = user to be modified.
 * cd->hostmask (optional) = user@host of user.
 * TODO: if owner && user == deleted, set deleted = 0.
 */
void process_cmd_adduser(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;

	ou = ocf_get_user(cd->name);

	/* User already exists. */
	if (ou) {
		chanfix_send_privmsg(md->nick, "User %s already exists.", ou->name);
		return;
	}

	ou = ocf_add_user(cd->name, cd->hostmask);
	if (ou) {
		strncpy(ou->password, "NOTSETYET", sizeof(ou->password));
		ocf_users_updated();
		if (cd->hostmask) {
			chanfix_send_privmsg(md->nick, "Created user %s (%s).", 
				cd->name, cd->hostmask);
			send_notice_to_loggedin_except(cd->loggedin_user_p, 
				OCF_NOTICE_FLAGS_USER,
				"%s added user %s (%s).", cd->loggedin_user_p->nick, ou->name,
				cd->hostmask);
			ocf_log(OCF_LOG_USERS, "USER %s added user %s (%s).", 
				cd->loggedin_user_p->nick, ou->name, cd->hostmask);
		} else {
			chanfix_send_privmsg(md->nick, "Created user %s.", cd->name);
			send_notice_to_loggedin(OCF_NOTICE_FLAGS_USER,
				"%s added user %s.", cd->loggedin_user_p->nick, ou->name);
			ocf_log(OCF_LOG_USERS, "USER %s added user %s.", 
				cd->loggedin_user_p->nick, ou->name);
		}
	} else {
		chanfix_send_privmsg(md->nick, "Error creating user %s.", cd->name);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_deluser(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;
	ou = ocf_get_user(cd->name);

	/* Test if the user does actually exist. */
	if (!ou) {
		chanfix_send_privmsg(md->nick, "No such user %s.", cd->name);
		return;
	}

	/* Can't delete an owner. */
	if (ocf_user_has_flag(ou, OCF_USER_FLAGS_OWNER)) {
		chanfix_send_privmsg(md->nick, "You cannot delete an owner.");
		return;
	}

	/* Can only delete a user manager if you're an owner. */
	if (ocf_user_has_flag(ou, OCF_USER_FLAGS_USERMANAGER) &&
		!ocf_user_has_flag(cd->loggedin_user_p->ocf_user_p, OCF_USER_FLAGS_OWNER)) {
		chanfix_send_privmsg(md->nick, "You cannot delete a user manager.");
		return;
	}

	ou->deleted = 1;
	chanfix_send_privmsg(md->nick, "Deleted user %s.", ou->name);
	ocf_users_updated();
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_USER,
		"%s deleted user %s.", cd->loggedin_user_p->nick, ou->name);
	ocf_log(OCF_LOG_USERS, "USER %s deleted user %s.", 
		cd->loggedin_user_p->nick, ou->name);
}

/* ----------------------------------------------------------------- */

void process_cmd_chpass(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;
	ou = ocf_get_user(cd->name);

	/* Test if the user does actually exist. */
	if (!ou) {
		chanfix_send_privmsg(md->nick, "No such user %s.", cd->name);
		return;
	}

	/* Can't change an owner's password unless you're an owner. */
	if (ocf_user_has_flag(ou, OCF_USER_FLAGS_OWNER) &&
		!ocf_user_has_flag(cd->loggedin_user_p->ocf_user_p, OCF_USER_FLAGS_OWNER)) {
		chanfix_send_privmsg(md->nick, "You cannot change an owner's password.");
		return;
	}

	ocf_user_set_password(ou, cd->password);;
	chanfix_send_privmsg(md->nick, "Changed %s's password.", ou->name);
	ocf_users_updated();
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_USER,
		"%s changed user %s's password.", cd->loggedin_user_p->nick, ou->name);
	ocf_log(OCF_LOG_USERS, "USER %s changed user %s's password.", 
		cd->loggedin_user_p->nick, ou->name);
}

/* ----------------------------------------------------------------- */

void process_cmd_who(struct message_data *md, struct command_data *cd)
{
	struct ocf_user *ou;

	dlink_node *ptr;
	struct ocf_loggedin_user *olu;
	char names[IRCLINELEN];
	ou = ocf_get_user(cd->name);
	names[0] = 0;

	chanfix_send_privmsg(md->nick, "Logged in users [nick (username:flags)]:");

	DLINK_FOREACH(ptr, ocf_loggedin_user_list.head) {
		olu = ((struct ocf_loggedin_user*)(ptr->data));
		strcat(names, olu->nick);
		strcat(names, " (");
		strcat(names, olu->ocf_user_p->name);
		if (olu->ocf_user_p->flags) {
			strcat(names, ":");
			strcat(names, ocf_user_flags_to_string(olu->ocf_user_p->flags));
		}
		strcat(names, ") ");
		
		/* If the string is getting too long, send it now and empty it. */
		if (strlen(names) > IRCLINELEN - 50) {
			chanfix_send_privmsg(md->nick, "%s", names);
			names[0] = 0;
		}
	}

	chanfix_send_privmsg(md->nick, "%s", names);
}

/* ----------------------------------------------------------------- */
/* UMODE [<+|->flags]
 * If the argument is + or - some flags, the flags are stored in
 * cd->flags. If +, then cd->params[0] = 1; If -, then cd->params[0] = 0. 
 * Without flags, cd->flags is 0. 
 */

void process_cmd_umode(struct message_data *md, struct command_data *cd)
{
	/* Update the umodes */
	if (cd->params[0] == '+') { 
		ocf_user_add_notice_flags(cd->loggedin_user_p->ocf_user_p, cd->flags);
		ocf_users_updated();
	}

	if (cd->params[0] == '-') { 
		ocf_user_del_notice_flags(cd->loggedin_user_p->ocf_user_p, cd->flags);
		ocf_users_updated();
	}

	/* Show the umodes */
	chanfix_send_privmsg(md->nick, "Your umodes are: %s.",
		ocf_user_notice_flags_to_string(cd->loggedin_user_p->ocf_user_p->noticeflags));
}

/* ----------------------------------------------------------------- */

void process_cmd_status(struct message_data *md, struct command_data *cd)
{
	char datetimestr[OCF_TIMESTAMP_LEN];
	struct tm *mlt;
	int num_servers_linked;

	mlt = localtime(&module_loaded_time);
	memset(datetimestr, 0, OCF_TIMESTAMP_LEN);
	strftime(datetimestr, OCF_TIMESTAMP_LEN, "%Y-%m-%d %H:%M:%S", mlt);

	chanfix_send_privmsg(md->nick, "This is OpenChanfix version %s (the %s release).",
		OCF_VERSION, OCF_VERSION_NAME);
	chanfix_send_privmsg(md->nick, "The module was loaded on %s. Status:", 
		datetimestr);

	chanfix_send_privmsg(md->nick, "Enable autofix is %d.", 
		settings_get_bool("enable_autofix"));
	chanfix_send_privmsg(md->nick, "Enable chanfix is %d.", 
		settings_get_bool("enable_chanfix"));
	chanfix_send_privmsg(md->nick, "Enable channel blocking is %d.", 
		settings_get_bool("enable_channel_blocking"));

	chanfix_send_privmsg(md->nick, 
		"Required amount of servers linked is %d percent of %d, which is a minimum of %d servers.", 
		settings_get_int("min_servers_present"),
		settings_get_int("num_servers"),
		(settings_get_int("min_servers_present") * 
			settings_get_int("num_servers")) / 100 + 1);

	autofix_disabled_due_to_split = !check_minimum_servers_linked();
	num_servers_linked = dlink_list_length(&global_serv_list);
	if (autofix_disabled_due_to_split) {
		if (num_servers_linked == 1) {
			chanfix_send_privmsg(md->nick, 
				"Chanfix is currently disabled because there are no servers linked.");
		} else {
			chanfix_send_privmsg(md->nick, 
				"Chanfix is currently disabled because only %d servers are linked.",
				num_servers_linked);
		}
	} else {
		chanfix_send_privmsg(md->nick, 
			"There are %d servers linked.", num_servers_linked);
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_set(struct message_data *md, struct command_data *cd)
{
	int val;
	if (!strcasecmp(cd->name, "num_servers")) {
		val = atoi(cd->params);
		if (val > 0) {
			settings_set_int("num_servers", val);
			chanfix_send_privmsg(md->nick, "NUM_SERVERS is now %d.",
				val);
			send_notice_to_loggedin_except(cd->loggedin_user_p, 
				OCF_NOTICE_FLAGS_MAIN,
				"%s changed NUM_SERVERS to %d.", 
				cd->loggedin_user_p->nick, val);
			ocf_log(OCF_LOG_MAIN, " SET %s changed NUM_SERVERS to %d.", 
				cd->loggedin_user_p->nick, val);
		} else {
			chanfix_send_privmsg(md->nick, 
				"Please use SET NUM_SERVERS <integer number>.");
		}
	}

	else if (!strcasecmp(cd->name, "enable_autofix")) {
		if (!strcasecmp(cd->params, "1") ||
				!strcasecmp(cd->params, "on")) {
			settings_set_bool("enable_autofix", 1);
			chanfix_send_privmsg(md->nick, "Enabled autofix.");
			send_notice_to_loggedin_except(cd->loggedin_user_p, 
				OCF_NOTICE_FLAGS_MAIN,
				"%s enabled autofix.", cd->loggedin_user_p->nick);
			ocf_log(OCF_LOG_MAIN, " SET %s enabled autofix.", 
				cd->loggedin_user_p->nick);
		} else if (!strcasecmp(cd->params, "0") ||
				!strcasecmp(cd->params, "off")) {
			settings_set_bool("enable_autofix", 0);
			chanfix_send_privmsg(md->nick, "Disabled autofix.");
			send_notice_to_loggedin_except(cd->loggedin_user_p, 
				OCF_NOTICE_FLAGS_MAIN,
				"%s disabled autofix.", cd->loggedin_user_p->nick);
			ocf_log(OCF_LOG_MAIN, " SET %s disabled autofix.", 
				cd->loggedin_user_p->nick);
		} else {
			chanfix_send_privmsg(md->nick, 
				"Please use SET ENABLE_AUTOFIX <on|off>.");
		}

	}

	else if (!strcasecmp(cd->name, "enable_chanfix")) {
		if (!strcasecmp(cd->params, "1") ||
				!strcasecmp(cd->params, "on")) {
			settings_set_bool("enable_chanfix", 1);
			chanfix_send_privmsg(md->nick, "Enabled manual chanfix.");
			send_notice_to_loggedin_except(cd->loggedin_user_p, 
				OCF_NOTICE_FLAGS_MAIN,
				"%s enabled manual chanfix.", cd->loggedin_user_p->nick);
			ocf_log(OCF_LOG_MAIN, " SET %s enabled manual chanfix.", 
				cd->loggedin_user_p->nick);
		} else if (!strcasecmp(cd->params, "0") ||
				!strcasecmp(cd->params, "off")) {
			settings_set_bool("enable_chanfix", 0);
			chanfix_send_privmsg(md->nick, "Disabled manual chanfix.");
			send_notice_to_loggedin_except(cd->loggedin_user_p, 
				OCF_NOTICE_FLAGS_MAIN,
				"%s disabled manual chanfix.", cd->loggedin_user_p->nick);
			ocf_log(OCF_LOG_MAIN, " SET %s disabled manual chanfix.", 
				cd->loggedin_user_p->nick);
		} else {
			chanfix_send_privmsg(md->nick, 
				"Please use SET ENABLE_CHANFIX <on|off>.");
		}

	} 
	
	else if (!strcasecmp(cd->name, "enable_channel_blocking")) {
		if (!strcasecmp(cd->params, "1") ||
				!strcasecmp(cd->params, "on")) {
			settings_set_bool("enable_channel_blocking", 1);
			chanfix_send_privmsg(md->nick, "Enabled channel blocking.");
			send_notice_to_loggedin_except(cd->loggedin_user_p, 
				OCF_NOTICE_FLAGS_MAIN,
				"%s enabled channel blocking.", cd->loggedin_user_p->nick);
			ocf_log(OCF_LOG_MAIN, " SET %s enabled channel blocking.", 
				cd->loggedin_user_p->nick);
		} else if (!strcasecmp(cd->params, "0") ||
				!strcasecmp(cd->params, "off")) {
			settings_set_bool("enable_channel_blocking", 0);
			chanfix_send_privmsg(md->nick, "Disabled channel blocking.");
			send_notice_to_loggedin_except(cd->loggedin_user_p, 
				OCF_NOTICE_FLAGS_MAIN,
				"%s disabled channel blocking.", cd->loggedin_user_p->nick);
			ocf_log(OCF_LOG_MAIN, " SET %s disabled channel blocking.", 
				cd->loggedin_user_p->nick);
		} else {
			chanfix_send_privmsg(md->nick, 
				"Please use SET ENABLE_CHANFIX <on|off>.");
		}

	} 
	
	else {
		chanfix_send_privmsg(md->nick, "This setting does not exist.");
	}
}

/* ----------------------------------------------------------------- */

void process_cmd_block(struct message_data *md, struct command_data *cd)
{
	unsigned int flags;
	char blocknote[IRCLINELEN];

	if (settings_get_bool("enable_channel_blocking") == 0) {
		chanfix_send_privmsg(md->nick, "Channel blocking is disabled.");
		return;
	}

	DB_set_read_channel(cd->channel);
	flags = DB_channel_get_flag();
	if (flags & OCF_CHANNEL_FLAG_BLOCK) {
		chanfix_send_privmsg(md->nick, "Channel %s is already blocked.",
			cd->channel);
		return;
	}
	
	sprintf(blocknote, "BLOCK by %s: %s", 
		cd->loggedin_user_p->ocf_user_p->name, cd->params);
	DB_set_write_channel(cd->channel);
	DB_channel_set_flag(OCF_CHANNEL_FLAG_BLOCK);
	DB_channel_add_note(cd->loggedin_user_p->ocf_user_p->id, time(NULL),
		blocknote);

	chanfix_send_privmsg(md->nick, "Channel %s has been blocked.",
		cd->channel);
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_BLOCK,
		"%s has blocked channel %s.", cd->loggedin_user_p->nick, cd->channel);
	ocf_log(OCF_LOG_CHANFIX, "BLCK %s has blocked channel %s.", 
		cd->loggedin_user_p->ocf_user_p->name, cd->channel);
}

/* ----------------------------------------------------------------- */

void process_cmd_unblock(struct message_data *md, struct command_data *cd)
{
	unsigned int flags;
	char unblocknote[IRCLINELEN];

	if (settings_get_bool("enable_channel_blocking") == 0) {
		chanfix_send_privmsg(md->nick, "Channel blocking is disabled.");
		return;
	}

	DB_set_read_channel(cd->channel);
	flags = DB_channel_get_flag();
	if (!(flags & OCF_CHANNEL_FLAG_BLOCK)) {
		chanfix_send_privmsg(md->nick, "Channel %s is not blocked.",
			cd->channel);
		return;
	}
	
	sprintf(unblocknote, "UNBLOCK by %s", cd->loggedin_user_p->ocf_user_p->name);
	DB_set_write_channel(cd->channel);
	DB_channel_set_flag(flags & ~OCF_CHANNEL_FLAG_BLOCK);
	DB_channel_add_note(cd->loggedin_user_p->ocf_user_p->id, time(NULL),
		unblocknote);

	chanfix_send_privmsg(md->nick, "Channel %s has been unblocked.",
		cd->channel);
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_BLOCK,
		"%s has unblocked channel %s.", cd->loggedin_user_p->nick, cd->channel);
	ocf_log(OCF_LOG_CHANFIX, "BLCK %s has unblocked channel %s.", 
		cd->loggedin_user_p->ocf_user_p->name, cd->channel);

	/* TODO: add the reason here, which is in cd->params */
}

/* ----------------------------------------------------------------- */

void process_cmd_rehash(struct message_data *md, struct command_data *cd)
{
	chanfix_send_privmsg(md->nick, "Rehashing.");
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_MAIN,
		"%s is rehashing.", cd->loggedin_user_p->nick);
	ocf_log(OCF_LOG_MAIN, "RHSH %s has rehashed.", 
		cd->loggedin_user_p->ocf_user_p->name);

	/* Save logged in users. */
	ocf_user_save_loggedin();

	/* Delete all users and logged in users. */
	ocf_users_deinit();
	ocf_logfiles_close();

	/* Rehash settings. */
	settings_read_file();
	ocf_logfiles_open();
	
	/* Load user file and restore logged in users. */
	ocf_user_load_all(settings_get_str("userfile"));
	ocf_user_load_loggedin();
}

/* ----------------------------------------------------------------- */

void process_cmd_check(struct message_data *md, struct command_data *cd)
{
	struct Channel *chptr;

	if (!settings_get_bool("enable_cmd_check")) {
		chanfix_send_privmsg(md->nick, "This command has been disabled.");
		return;
	}

	chptr = hash_find_channel(cd->channel);

	if (!chptr) {
		chanfix_send_privmsg(md->nick, "No such channel %s.", cd->channel);
		return;
	}

	/* Reports ops and total clients. */
	chanfix_send_privmsg(md->nick, 
		"I see %d opped out of %d total clients in \"%s\".",
		ocf_get_number_chanops(chptr),
		dlink_list_length(&chptr->members), cd->channel);

	/* Log command. */
	if (cd->loggedin_user_p) {
		send_notice_to_loggedin_except(cd->loggedin_user_p, 
			OCF_NOTICE_FLAGS_CHANFIX,
			"%s has requested a CHECK on channel \"%s\"", cd->loggedin_user_p->nick,
			cd->channel);
	} else {
		send_notice_to_loggedin(OCF_NOTICE_FLAGS_CHANFIX,
			"%s has requested a CHECK on channel \"%s\"", cd->loggedin_user_p->nick,
			cd->channel);
	}
	ocf_log(OCF_LOG_CHANFIX, "CHCK %s has requested check %s", 
		cd->loggedin_user_p->ocf_user_p->name, cd->channel);
}

/* ----------------------------------------------------------------- */

void process_cmd_opnicks(struct message_data *md, struct command_data *cd)
{
	struct Channel *chptr;
	int num_ops = 0;
	char line[IRCLINELEN + 1];
	dlink_node *node;
	struct Membership* mship;

	if (!settings_get_bool("enable_cmd_opnicks")) {
		chanfix_send_privmsg(md->nick, "This command has been disabled.");
		return;
	}

	chptr = hash_find_channel(cd->channel);

	if (!chptr) {
		chanfix_send_privmsg(md->nick, "No such channel %s.", cd->channel);
		return;
	}

	/* Send list of opped clients. */
	chanfix_send_privmsg(md->nick, "Opped clients on channel \"%s\":",
		cd->channel);

	line[0] = '\0';

	DLINK_FOREACH(node, chptr->members.head) {
		mship = (struct Membership*)(node->data);
		if (mship->flags & CHFL_CHANOP) {
			num_ops++;
			strcat(line, mship->client_p->name);
			strcat(line, " ");
			if (strlen(line) > IRCLINELEN / 2) {
				/* Send line and clear the line. */
				chanfix_send_privmsg(md->nick, "%s", line);
				line[0] = '\0';
			}
		}
	}

	if (strlen(line) > 0) {
		chanfix_send_privmsg(md->nick, "%s", line);
	}

	if (num_ops == 1) {
		chanfix_send_privmsg(md->nick, "I see 1 opped client in \"%s\".", 
			cd->channel);
	} else {
		chanfix_send_privmsg(md->nick, "I see %d opped clients in \"%s\".", 
			num_ops, cd->channel);
	}

	/* Log command. */
	if (cd->loggedin_user_p) {
		send_notice_to_loggedin_except(cd->loggedin_user_p, 
			OCF_NOTICE_FLAGS_CHANFIX,
			"%s has requested OPNICKS on channel \"%s\"", cd->loggedin_user_p->nick,
			cd->channel);
	} else {
		send_notice_to_loggedin(OCF_NOTICE_FLAGS_CHANFIX,
			"%s has requested OPNICKS on channel \"%s\"", cd->loggedin_user_p->nick,
			cd->channel);
	}
	ocf_log(OCF_LOG_CHANFIX, "OPNK %s has requested OPNICKS %s", 
		cd->loggedin_user_p->ocf_user_p->name, cd->channel);
}

/* ----------------------------------------------------------------- */

void process_cmd_alert(struct message_data *md, struct command_data *cd)
{
	unsigned int flags;
	char blocknote[IRCLINELEN];

	DB_set_read_channel(cd->channel);
	flags = DB_channel_get_flag();
	if (flags & OCF_CHANNEL_FLAG_ALERT) {
		chanfix_send_privmsg(md->nick, "Channel %s already has the ALERT flag.",
			cd->channel);
		return;
	}
	
	sprintf(blocknote, "ALERT flag added by %s", 
		cd->loggedin_user_p->ocf_user_p->name);
	DB_set_write_channel(cd->channel);
	DB_channel_set_flag(OCF_CHANNEL_FLAG_ALERT);
	DB_channel_add_note(cd->loggedin_user_p->ocf_user_p->id, time(NULL),
		blocknote);

	chanfix_send_privmsg(md->nick, "ALERT flag added to channel \"%s\".",
		cd->channel);
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_CHANFIX,
		"%s has alerted channel %s.", cd->loggedin_user_p->nick, cd->channel);
	ocf_log(OCF_LOG_CHANFIX, "ALRT %s has alerted channel %s", 
		cd->loggedin_user_p->ocf_user_p->name, cd->channel);
}

/* ----------------------------------------------------------------- */

void process_cmd_unalert(struct message_data *md, struct command_data *cd)
{
	unsigned int flags;
	char unblocknote[IRCLINELEN];

	DB_set_read_channel(cd->channel);
	flags = DB_channel_get_flag();
	if (!(flags & OCF_CHANNEL_FLAG_ALERT)) {
		chanfix_send_privmsg(md->nick, "Channel %s does not have the ALERT flag.",
			cd->channel);
		return;
	}
	
	sprintf(unblocknote, "UNALERT by %s", cd->loggedin_user_p->ocf_user_p->name);
	DB_set_write_channel(cd->channel);
	DB_channel_set_flag(flags & ~OCF_CHANNEL_FLAG_ALERT);
	DB_channel_add_note(cd->loggedin_user_p->ocf_user_p->id, time(NULL),
		unblocknote);

	chanfix_send_privmsg(md->nick, "ALERT flag removed from channel \"%s\".",
		cd->channel);
	send_notice_to_loggedin_except(cd->loggedin_user_p, 
		OCF_NOTICE_FLAGS_CHANFIX,
		"%s has unalerted channel %s.", cd->loggedin_user_p->nick, cd->channel);
	ocf_log(OCF_LOG_CHANFIX, "ALRT %s has unalerted channel %s.", 
		cd->loggedin_user_p->ocf_user_p->name, cd->channel);
}

/* ----------------------------------------------------------------- */

